package com.example.sefinsa_app;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.View;
import android.widget.TextView;

import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.toolbox.JsonObjectRequest;
import com.example.sefinsa_app.api.API;
import com.example.sefinsa_app.utilities.ErrorChecker;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

public class NetworkSpeedChecker {

    private final Context context;
    private final TextView textViewMensajes;
    private final ScheduledExecutorService scheduler;
    private final Handler handler;
    private boolean wasSlowNetworkPreviously = false;
    private ScheduledFuture<?> scheduledFuture;
    private boolean isRequestInProgress = false; // Para controlar la concurrencia

    public NetworkSpeedChecker(Context context, TextView textViewMensajes) {
        this.context = context;
        this.textViewMensajes = textViewMensajes;
        this.scheduler = Executors.newScheduledThreadPool(1);
        this.handler = new Handler(Looper.getMainLooper());
    }

    public void startChecking() {
        if (scheduledFuture == null || scheduledFuture.isCancelled()) {
            scheduledFuture = scheduler.scheduleAtFixedRate(this::checkNetworkSpeed, 0, 60, TimeUnit.SECONDS);
        }
    }

    public void stopChecking() {
        if (scheduledFuture != null && !scheduledFuture.isCancelled()) {
            scheduledFuture.cancel(true);
        }
    }

    private synchronized void checkNetworkSpeed() {
        if (isRequestInProgress) {
            //Log.d("NETWORK_SPEED", "Solicitud anterior aún en progreso, esperando...");
            return;
        }

        isRequestInProgress = true;
        long startTime = System.currentTimeMillis();

        JSONObject data = new JSONObject();
        try {
            data.put("func", "index");
            data.put("estatus", "0");
            //Log.d("NETWORK_SPEED", "Parametros enviados NETWORKCHECKER: " + data.toString());
        } catch (JSONException e) {
            e.printStackTrace();
            isRequestInProgress = false;
            return;
        }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlPrestamos, data,
                response -> {
                    long endTime = System.currentTimeMillis();
                    long duration = endTime - startTime;

                    //Log.d("NETWORK_SPEED", "Duración de la solicitud: " + duration + " ms");

                    if (duration > 5000) {
                        if (!wasSlowNetworkPreviously) {
                            showSlowNetworkMessage();
                            wasSlowNetworkPreviously = true;
                        }
                    } else {
                        if (wasSlowNetworkPreviously) {
                            showGoodNetworkMessage();
                            wasSlowNetworkPreviously = false;
                        }
                    }
                    isRequestInProgress = false; // Marca la solicitud como completada
                },
                error -> {
                    long endTime = System.currentTimeMillis();
                    long duration = endTime - startTime;

                    //Log.e("NETWORK_SPEED", "Error en la solicitud después de " + duration + " ms: " + error.getMessage());

                    ErrorChecker.checker(error, context);
                    isRequestInProgress = false; // Marca la solicitud como completada incluso en caso de error
                });

        request.setRetryPolicy(new DefaultRetryPolicy(
                30000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }

    private void showSlowNetworkMessage() {
        //Log.d("NETWORK_SPEED", "Mostrando mensaje de red lenta o intermitente.");
        handler.post(() -> {
            if (textViewMensajes != null) {
                //Log.d("NETWORK_SPEED", "TextView encontrado, mostrando mensaje.");
                textViewMensajes.setVisibility(View.VISIBLE);
                textViewMensajes.setTextColor(context.getResources().getColor(R.color.yellow));
                textViewMensajes.setTextSize(14);
                textViewMensajes.setText("¡Red lenta o intermitente!");
            } else {
                Log.e("NETWORK_SPEED", "TextView es nulo.");
            }
        });
    }

    private void showGoodNetworkMessage() {
        //Log.d("NETWORK_SPEED", "Ocultando mensaje de red lenta.");
        handler.post(() -> {
            if (textViewMensajes != null) {
                //Log.d("NETWORK_SPEED", "TextView encontrado, ocultando mensaje.");
                textViewMensajes.setVisibility(View.VISIBLE);
                textViewMensajes.setText("");
            } else {
                Log.e("NETWORK_SPEED", "TextView es nulo.");
            }
        });
    }
}


